'use client';

import { useParams, useRouter } from 'next/navigation';
import { Card, CardContent, CardHeader, CardTitle, CardDescription, CardFooter } from '@/components/ui/card';
import { Alert, AlertDescription, AlertTitle } from '@/components/ui/alert';
import { User as UserIcon, ArrowLeft, Mail, Phone, Calendar, UserX, Trash2, LoaderCircle, History, MapPin, KeyRound, UserCheck } from 'lucide-react';
import { Button } from '@/components/ui/button';
import { useDoc, useFirestore, useMemoFirebase, useAuth } from '@/firebase';
import { doc, updateDoc, deleteDoc } from 'firebase/firestore';
import type { User } from '@/lib/data';
import { Skeleton } from '@/components/ui/skeleton';
import { Avatar, AvatarFallback, AvatarImage } from '@/components/ui/avatar';
import {
  AlertDialog,
  AlertDialogAction,
  AlertDialogCancel,
  AlertDialogContent,
  AlertDialogDescription,
  AlertDialogFooter,
  AlertDialogHeader,
  AlertDialogTitle,
  AlertDialogTrigger,
} from '@/components/ui/alert-dialog';
import { useToast } from '@/hooks/use-toast';
import { useState } from 'react';
import { Separator } from '@/components/ui/separator';
import Link from 'next/link';
import { sendPasswordResetEmail } from 'firebase/auth';
import { errorEmitter } from '@/firebase/error-emitter';
import { FirestorePermissionError } from '@/firebase/errors';

export default function UserDetailsPage() {
  const params = useParams();
  const router = useRouter();
  const userId = params.userId as string;
  const firestore = useFirestore();
  const auth = useAuth();
  const { toast } = useToast();
  const [isDeleting, setIsDeleting] = useState(false);
  const [isSuspending, setIsSuspending] = useState(false);

  const userDocRef = useMemoFirebase(
    () => (userId ? doc(firestore, 'users', userId) : null),
    [firestore, userId]
  );
  
  const { data: user, isLoading, error } = useDoc<User>(userDocRef);

  const getInitials = (name: string) => {
    return name?.split(' ').map(n => n[0]).join('').toUpperCase() || '';
  }

  const registeredDate = user?.registeredDate
    ? new Date(user.registeredDate.seconds * 1000).toLocaleDateString()
    : 'N/A';

  const handleResetPassword = async () => {
    if (!user?.email) {
       toast({
        variant: 'destructive',
        title: 'Action Failed',
        description: 'User email is not available to send a reset link.',
      });
      return;
    }
    try {
      await sendPasswordResetEmail(auth, user.email);
      toast({
        title: 'Password Reset Email Sent',
        description: `An email has been sent to ${user.email} with instructions to reset their password.`,
      });
    } catch (error: any) {
       toast({
        variant: 'destructive',
        title: 'Action Failed',
        description: 'Could not send password reset email. Please try again.',
      });
    }
  }

  const handleToggleSuspend = async () => {
    if (!userDocRef) return;
    setIsSuspending(true);
    const newSuspendedState = !user?.suspended;
    
    updateDoc(userDocRef, { suspended: newSuspendedState })
        .then(() => {
            toast({
                title: newSuspendedState ? 'User Suspended' : 'User Unsuspended',
                description: `${user?.name} has been ${newSuspendedState ? 'suspended' : 're-activated'}.`,
            });
        })
        .catch(() => {
            errorEmitter.emit(
                'permission-error',
                new FirestorePermissionError({
                    path: userDocRef.path,
                    operation: 'update',
                    requestResourceData: { suspended: newSuspendedState },
                })
            );
        })
        .finally(() => {
            setIsSuspending(false);
        });
  };

  const handleDelete = async () => {
    if (!userDocRef) return;
    setIsDeleting(true);
    
    deleteDoc(userDocRef)
      .then(() => {
        toast({
          title: 'User Data Deleted',
          description: `All Firestore data for ${user?.name} has been removed.`,
        });
        router.push('/admin/users');
      })
      .catch((e) => {
        errorEmitter.emit(
          'permission-error',
          new FirestorePermissionError({
            path: userDocRef.path,
            operation: 'delete',
          })
        );
      })
      .finally(() => {
        setIsDeleting(false);
      });
  }

  return (
    <div className="p-4 md:p-8">
      <div className="flex items-center mb-4">
        <Button variant="ghost" size="icon" onClick={() => router.back()} className="mr-2">
            <ArrowLeft className="h-4 w-4" />
        </Button>
        <h1 className="text-2xl font-bold">User Details</h1>
      </div>
      
      {isLoading && (
        <Card>
          <CardHeader>
            <Skeleton className="h-8 w-48" />
            <Skeleton className="h-4 w-64 mt-2" />
          </CardHeader>
          <CardContent>
            <div className="grid md:grid-cols-3 gap-6">
                <div className="flex flex-col items-center md:items-start gap-4">
                  <Skeleton className="h-24 w-24 rounded-full" />
                </div>
                <div className="md:col-span-2 grid grid-cols-1 sm:grid-cols-2 gap-y-6 gap-x-8 pt-2">
                   <Skeleton className="h-10 w-full" />
                   <Skeleton className="h-10 w-full" />
                   <Skeleton className="h-10 w-full" />
                </div>
            </div>
          </CardContent>
        </Card>
      )}

      {!isLoading && error && (
         <Card>
          <CardContent className="pt-6">
            <Alert variant="destructive">
              <UserIcon className="h-4 w-4" />
              <AlertTitle>User Not Found</AlertTitle>
              <AlertDescription>
                The user you are looking for does not exist or there was an error fetching their data.
              </AlertDescription>
            </Alert>
          </CardContent>
        </Card>
      )}

      {!isLoading && !error && user && (
        <Card>
            <CardHeader>
                <CardTitle>User Information</CardTitle>
                <CardDescription>Details for user #{user.id}</CardDescription>
            </CardHeader>
            <CardContent>
                 <div className="grid md:grid-cols-3 gap-6">
                    <div className="flex flex-col items-center md:items-start gap-4">
                        <Avatar className="h-24 w-24 text-3xl">
                            <AvatarImage src={user.photoURL} alt={user.name} />
                            <AvatarFallback>{getInitials(user.name)}</AvatarFallback>
                        </Avatar>
                    </div>
                    <div className="md:col-span-2 grid grid-cols-1 sm:grid-cols-2 gap-y-6 gap-x-8 pt-2">
                        <div>
                            <p className="text-sm text-muted-foreground">Full Name</p>
                            <p className="font-medium flex items-center gap-2 mt-1"><UserIcon className="w-4 h-4 text-primary" /> {user.name}</p>
                        </div>
                         <div>
                            <p className="text-sm text-muted-foreground">Email Address</p>
                            <p className="font-medium flex items-center gap-2 mt-1"><Mail className="w-4 h-4 text-primary" /> {user.email}</p>
                        </div>
                        <div>
                            <p className="text-sm text-muted-foreground">Registered On</p>
                            <p className="font-medium flex items-center gap-2 mt-1"><Calendar className="w-4 h-4 text-primary" /> {registeredDate}</p>
                        </div>
                    </div>
                </div>
            </CardContent>
            <CardFooter className="border-t pt-6 flex flex-col items-start gap-4">
                <div className="flex flex-wrap gap-2">
                     <Button asChild variant="outline">
                        <Link href={`/admin/users/${userId}/orders`}>
                            <History className="mr-2" />
                            Order History
                        </Link>
                    </Button>
                    <Button asChild variant="outline">
                        <Link href={`/admin/users/${userId}/addresses`}>
                           <MapPin className="mr-2" />
                           Address List
                        </Link>
                    </Button>
                    <Button variant="outline" onClick={handleResetPassword}>
                        <KeyRound className="mr-2" />
                        Reset Password
                    </Button>
                </div>
                <Separator />
                 <div className="flex flex-wrap gap-2">
                    <Button
                        variant="outline"
                        className={user.suspended ? "text-green-600 border-green-600/50 hover:bg-green-100/50 hover:text-green-700" : "text-amber-600 border-amber-600/50 hover:bg-amber-100/50 hover:text-amber-700"}
                        onClick={handleToggleSuspend}
                        disabled={isSuspending}
                    >
                        {isSuspending ? (
                            <LoaderCircle className="mr-2 h-4 w-4 animate-spin" />
                        ) : user.suspended ? (
                            <UserCheck className="mr-2" />
                        ) : (
                            <UserX className="mr-2" />
                        )}
                        {user.suspended ? 'Unsuspend User' : 'Suspend User'}
                    </Button>

                    <AlertDialog>
                      <AlertDialogTrigger asChild>
                        <Button variant="destructive">
                            <Trash2 className="mr-2" />
                            Delete User Data
                        </Button>
                      </AlertDialogTrigger>
                      <AlertDialogContent>
                        <AlertDialogHeader>
                          <AlertDialogTitle>Are you sure?</AlertDialogTitle>
                          <AlertDialogDescription>
                            This action will permanently delete the user's Firestore data (profile, orders, etc.), but will not delete their authentication account. This is a client-side action; a secure backend function is required to fully delete a user.
                          </AlertDialogDescription>
                        </AlertDialogHeader>
                        <AlertDialogFooter>
                          <AlertDialogCancel>Cancel</AlertDialogCancel>
                          <AlertDialogAction onClick={handleDelete} disabled={isDeleting} className="bg-destructive hover:bg-destructive/90">
                             {isDeleting && <LoaderCircle className="animate-spin mr-2 h-4 w-4" />}
                            Delete Data
                          </AlertDialogAction>
                        </AlertDialogFooter>
                      </AlertDialogContent>
                    </AlertDialog>
                </div>
            </CardFooter>
        </Card>
      )}
    </div>
  );
}
